// Admin Dashboard JavaScript
let events = [];
let currentEditId = null;
let deleteEventId = null;

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    checkAuth();
    loadEvents();
    updateStats();
});

// Check authentication
function checkAuth() {
    const sessionId = localStorage.getItem('adminSession');
    const username = localStorage.getItem('adminUsername');
    
    if (!sessionId) {
        window.location.href = 'login.html';
        return;
    }
    
    // Update username display
    if (username) {
        document.getElementById('username').textContent = username;
    }
    
    // Verify session
    fetch('api/check_session.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            session_id: sessionId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (!data.success) {
            localStorage.removeItem('adminSession');
            localStorage.removeItem('adminUsername');
            window.location.href = 'login.html';
        }
    })
    .catch(error => {
        console.error('Session check error:', error);
        window.location.href = 'login.html';
    });
}

// Load events
async function loadEvents() {
    const sessionId = localStorage.getItem('adminSession');
    
    try {
        const response = await fetch('api/events.php', {
            method: 'GET',
            headers: {
                'Authorization': 'Bearer ' + sessionId
            }
        });
        
        const data = await response.json();
        
        if (data.success) {
            events = data.events;
            renderEvents();
            updateStats();
        } else {
            showToast('Error loading events: ' + data.message, 'error');
        }
    } catch (error) {
        console.error('Load events error:', error);
        showToast('Error loading events', 'error');
    }
}

// Render events table
function renderEvents(filteredEvents = null) {
    const tbody = document.getElementById('eventsTableBody');
    const eventsToRender = filteredEvents || events;
    
    if (eventsToRender.length === 0) {
        tbody.innerHTML = '<tr><td colspan="8" style="text-align: center; padding: 2rem;">No events found</td></tr>';
        return;
    }
    
    tbody.innerHTML = eventsToRender.map(event => `
        <tr>
            <td>${event.id}</td>
            <td>${getMonthName(event.month)}</td>
            <td>${event.day}</td>
            <td><strong>${escapeHtml(event.title)}</strong></td>
            <td>${escapeHtml(event.description || '')}</td>
            <td>
                <span class="status ${event.is_active ? 'active' : 'inactive'}">
                    ${event.is_active ? 'Active' : 'Inactive'}
                </span>
            </td>
            <td>${formatDate(event.created_at)}</td>
            <td>
                <div class="actions">
                    <button class="btn btn-sm btn-warning" onclick="editEvent(${event.id})">Edit</button>
                    <button class="btn btn-sm btn-danger" onclick="deleteEvent(${event.id}, '${escapeHtml(event.title)}')">Delete</button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Update statistics
function updateStats() {
    const total = events.length;
    const active = events.filter(e => e.is_active == 1).length;
    const inactive = events.filter(e => e.is_active == 0).length;
    const currentMonth = new Date().getMonth() + 1;
    const thisMonth = events.filter(e => e.month == currentMonth).length;
    
    document.getElementById('totalEvents').textContent = total;
    document.getElementById('activeEvents').textContent = active;
    document.getElementById('inactiveEvents').textContent = inactive;
    document.getElementById('thisMonthEvents').textContent = thisMonth;
}

// Search events
function searchEvents() {
    const searchTerm = document.getElementById('searchInput').value.toLowerCase();
    
    if (!searchTerm) {
        renderEvents();
        return;
    }
    
    const filtered = events.filter(event => 
        event.title.toLowerCase().includes(searchTerm) ||
        (event.description && event.description.toLowerCase().includes(searchTerm)) ||
        event.month.toString().includes(searchTerm) ||
        event.day.toString().includes(searchTerm)
    );
    
    renderEvents(filtered);
}

// Open add modal
function openAddModal() {
    currentEditId = null;
    document.getElementById('modalTitle').textContent = 'Add Event';
    document.getElementById('eventForm').reset();
    document.getElementById('eventModal').classList.add('active');
}

// Edit event
function editEvent(id) {
    const event = events.find(e => e.id === id);
    if (!event) return;
    
    currentEditId = id;
    document.getElementById('modalTitle').textContent = 'Edit Event';
    document.getElementById('eventId').value = event.id;
    document.getElementById('month').value = event.month;
    document.getElementById('day').value = event.day;
    document.getElementById('title').value = event.title;
    document.getElementById('description').value = event.description || '';
    document.getElementById('is_active').value = event.is_active;
    
    document.getElementById('eventModal').classList.add('active');
}

// Save event
async function saveEvent() {
    const sessionId = localStorage.getItem('adminSession');
    const formData = {
        month: parseInt(document.getElementById('month').value),
        day: parseInt(document.getElementById('day').value),
        title: document.getElementById('title').value.trim(),
        description: document.getElementById('description').value.trim(),
        is_active: parseInt(document.getElementById('is_active').value)
    };
    
    // Validation
    if (!formData.month || !formData.day || !formData.title) {
        showToast('Please fill all required fields', 'error');
        return;
    }
    
    if (formData.day < 1 || formData.day > 31) {
        showToast('Day must be between 1 and 31', 'error');
        return;
    }
    
    // Show loading
    document.getElementById('saveBtnText').style.display = 'none';
    document.getElementById('saveBtnLoading').style.display = 'inline-block';
    
    try {
        const url = currentEditId ? 'api/events.php?id=' + currentEditId : 'api/events.php';
        const method = currentEditId ? 'PUT' : 'POST';
        
        const response = await fetch(url, {
            method: method,
            headers: {
                'Content-Type': 'application/json',
                'Authorization': 'Bearer ' + sessionId
            },
            body: JSON.stringify(formData)
        });
        
        const data = await response.json();
        
        if (data.success) {
            showToast(currentEditId ? 'Event updated successfully' : 'Event added successfully', 'success');
            closeModal();
            loadEvents();
        } else {
            showToast('Error: ' + data.message, 'error');
        }
    } catch (error) {
        console.error('Save event error:', error);
        showToast('Error saving event', 'error');
    } finally {
        document.getElementById('saveBtnText').style.display = 'inline';
        document.getElementById('saveBtnLoading').style.display = 'none';
    }
}

// Delete event
function deleteEvent(id, title) {
    deleteEventId = id;
    document.getElementById('deleteEventTitle').textContent = title;
    document.getElementById('deleteModal').classList.add('active');
}

// Confirm delete
async function confirmDelete() {
    const sessionId = localStorage.getItem('adminSession');
    
    document.getElementById('deleteBtnText').style.display = 'none';
    document.getElementById('deleteBtnLoading').style.display = 'inline-block';
    
    try {
        const response = await fetch('api/events.php?id=' + deleteEventId, {
            method: 'DELETE',
            headers: {
                'Authorization': 'Bearer ' + sessionId
            }
        });
        
        const data = await response.json();
        
        if (data.success) {
            showToast('Event deleted successfully', 'success');
            closeDeleteModal();
            loadEvents();
        } else {
            showToast('Error: ' + data.message, 'error');
        }
    } catch (error) {
        console.error('Delete event error:', error);
        showToast('Error deleting event', 'error');
    } finally {
        document.getElementById('deleteBtnText').style.display = 'inline';
        document.getElementById('deleteBtnLoading').style.display = 'none';
    }
}

// Close modal
function closeModal() {
    document.getElementById('eventModal').classList.remove('active');
    document.getElementById('eventForm').reset();
    currentEditId = null;
}

// Close delete modal
function closeDeleteModal() {
    document.getElementById('deleteModal').classList.remove('active');
    deleteEventId = null;
}

// Logout
function logout() {
    const sessionId = localStorage.getItem('adminSession');
    
    if (sessionId) {
        fetch('api/logout.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                session_id: sessionId
            })
        })
        .then(response => response.json())
        .then(data => {
            console.log('Logout response:', data);
        })
        .catch(error => {
            console.error('Logout error:', error);
        });
    }
    
    localStorage.removeItem('adminSession');
    localStorage.removeItem('adminUsername');
    window.location.href = 'login.html';
}

// Utility functions
function getMonthName(month) {
    const months = ['', 'January', 'February', 'March', 'April', 'May', 'June',
                   'July', 'August', 'September', 'October', 'November', 'December'];
    return months[month] || month;
}

function formatDate(dateString) {
    if (!dateString) return '-';
    const date = new Date(dateString);
    return date.toLocaleDateString('id-ID', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

function escapeHtml(text) {
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
}

function showToast(message, type = 'info') {
    const toast = document.createElement('div');
    toast.className = `toast ${type}`;
    toast.textContent = message;
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.style.animation = 'slideInRight 0.3s ease reverse';
        setTimeout(() => {
            document.body.removeChild(toast);
        }, 300);
    }, 3000);
}

// Close modals on ESC key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeModal();
        closeDeleteModal();
    }
});

// Close modals on outside click
document.getElementById('eventModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeModal();
    }
});

document.getElementById('deleteModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeDeleteModal();
    }
});